<?php

$categoriesObj = new categories();

class categories
{
    var $mDb;
    var $mConfig;
    var $mlang;

    function __construct()
    {
        global $Config;
        global $Lang;

        $this->mDb = new iplus();
        $this->mConfig = $Config;
        $this->mlang = $Lang;
    }


    function convert_object_to_array($data)
    {
        if (is_object($data)) {
            $data = get_object_vars($data);
        }

        if (is_array($data)) {
            return array_map(__METHOD__, $data);
        } else {
            return $data;
        }
    }

    function getAllCategoriesWithLang($lang_code = 'ar')
    {
        try {
            $sql = "SELECT 
                c.id,
                c.status,
                c.image,
                c.parent_id,
                c.date_added,
                cl.name,
                cl.description,
                cl.lang_code
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            ORDER BY c.parent_id ASC, c.id ASC";

            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);
            $stmt->execute();

            $result = $stmt->fetchAll(PDO::FETCH_OBJ);
            return $this->convert_object_to_array($result);
        } catch (PDOException $e) {
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }

    function getSearchCategories($searchName = '', $searchId = '', $searchParentId = '', $searchSort = '', $lang_code = 'ar', $aStart = 0, $aLimit = 10, $type = 'ASC')
    {
        // التحقق من صحة نوع الترتيب (type)
        $type = strtoupper($type);
        if (!in_array($type, ['ASC', 'DESC'])) {
            $type = 'ASC';
        }

        $sql = "SELECT 
                c.id,
                c.status,
                c.image,
                c.parent_id,
                c.sort,
                cl.name,
                cl.description
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE 1";

        if (!empty($searchName)) {
            $sql .= " AND cl.name LIKE :searchName";
        }
        if (!empty($searchId)) {
            $sql .= " AND c.id = :searchId";
        }
        if (!empty($searchParentId)) {
            $sql .= " AND c.parent_id = :searchParentId";
        }
        if (!empty($searchSort)) {
            $sql .= " AND c.sort = :searchSort";
        }

        // ترتيب البيانات
        $sql .= " ORDER BY c.sort {$type}, c.id {$type}";

        // إضافة Limit لو فيه بيانات
        if (!is_null($aStart) && !is_null($aLimit)) {
            $sql .= " LIMIT :aStart, :aLimit";
        }

        try {
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);

            if (!empty($searchName)) {
                $searchName = "%{$searchName}%";
                $stmt->bindParam(':searchName', $searchName, PDO::PARAM_STR);
            }
            if (!empty($searchId)) {
                $stmt->bindParam(':searchId', $searchId, PDO::PARAM_INT);
            }
            if (!empty($searchParentId)) {
                $stmt->bindParam(':searchParentId', $searchParentId, PDO::PARAM_INT);
            }
            if (!empty($searchSort)) {
                $stmt->bindParam(':searchSort', $searchSort, PDO::PARAM_INT);
            }

            if (!is_null($aStart) && !is_null($aLimit)) {
                $stmt->bindParam(':aStart', $aStart, PDO::PARAM_INT);
                $stmt->bindParam(':aLimit', $aLimit, PDO::PARAM_INT);
            }

            $stmt->execute();
            $result = $stmt->fetchAll(PDO::FETCH_OBJ);
            return $this->convert_object_to_array($result);
        } catch (PDOException $e) {
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }

    function deleteCategoryWithChildren($category_id)
    {
        try {
            $db = $this->mDb->connect();
            $db->beginTransaction();

            $category_id = intval($category_id);
            if ($category_id <= 0) {
                return ['status' => 400, 'message' => 'Invalid category ID'];
            }

            // دالة لتجميع كل الأقسام الفرعية + الأصل
            $collectChildIds = function ($parent_id) use ($db, &$collectChildIds) {
                $ids = [];

                $stmt = $db->prepare("SELECT id FROM categories WHERE parent_id = :parent_id");
                $stmt->execute([':parent_id' => $parent_id]);
                $children = $stmt->fetchAll(PDO::FETCH_COLUMN);

                foreach ($children as $childId) {
                    $ids[] = $childId;
                    $ids = array_merge($ids, $collectChildIds($childId));
                }

                return $ids;
            };

            // جمع IDs
            $allChildren = $collectChildIds($category_id);
            $allIds = array_merge([$category_id], $allChildren);

            // 📸 حذف الصور من السيرفر
            $placeholders = implode(',', array_fill(0, count($allIds), '?'));
            $stmtImgs = $db->prepare("SELECT image FROM categories WHERE id IN ($placeholders)");
            $stmtImgs->execute($allIds);
            $images = $stmtImgs->fetchAll(PDO::FETCH_COLUMN);

            $uploadPath = __DIR__ . '/../../uploads/categories/';
            foreach ($images as $img) {
                if ($img && file_exists($uploadPath . $img)) {
                    unlink($uploadPath . $img);
                }
            }

            // 🗑 حذف الترجمات
            $stmtLang = $db->prepare("DELETE FROM categories_langs WHERE id_category IN ($placeholders)");
            $stmtLang->execute($allIds);

            // 🗑 حذف من جدول الأقسام
            $stmtCat = $db->prepare("DELETE FROM categories WHERE id IN ($placeholders)");
            $stmtCat->execute($allIds);

            $db->commit();

            return [
                'status' => 200,
                'message' => 'Category and subcategories deleted successfully',
                'deleted_ids' => $allIds
            ];
        } catch (PDOException $e) {
            $db->rollBack();
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }


    function getSearchCategoriesCount($searchName = '', $searchId = '', $searchParentId = '', $searchSort = '', $lang_code = 'ar'){
        $lang_code = LANGUAGE;

        $sql = "SELECT COUNT(c.id) as count
            FROM categories c
            LEFT JOIN categories_langs cl ON c.id = cl.id_category AND cl.lang_code = :lang_code
            WHERE 1 ";

        // الشروط حسب ما المستخدم يرسل
        if (!empty($searchName)) {
            $sql .= " AND cl.name LIKE :searchName";
        }
        if (!empty($searchId)) {
            $sql .= " AND c.id = :searchId";
        }
        if (!empty($searchParentId)) {
            $sql .= " AND c.parent_id = :searchParentId";
        }
        if (!empty($searchSort)) {
            $sql .= " AND c.sort = :searchSort";
        }

        try {
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);

            // ربط الباراميترز لو تم إرسالهم
            if (!empty($searchName)) {
                $searchName = '%' . $searchName . '%';
                $stmt->bindParam(':searchName', $searchName, PDO::PARAM_STR);
            }
            if (!empty($searchId)) {
                $stmt->bindParam(':searchId', $searchId, PDO::PARAM_INT);
            }
            if (!empty($searchParentId)) {
                $stmt->bindParam(':searchParentId', $searchParentId, PDO::PARAM_INT);
            }
            if (!empty($searchSort)) {
                $stmt->bindParam(':searchSort', $searchSort, PDO::PARAM_INT);
            }

            $stmt->execute();
            return $stmt->fetchColumn();
        } catch (PDOException $e) {
            error_log("PDOException in getSearchCategoriesCount: " . $e->getMessage());
            return 0;
        }
    }


    function getMainCategories($searchName = '', $searchId = '', $searchParentId = '', $searchSort = '', $lang_code = 'ar', $aStart = 0, $aLimit = 10, $type = 'ASC')
    {
        // $lang_code = LANGUAGE;
        $type = strtoupper($type);
        if (!in_array($type, ['ASC', 'DESC'])) {
            $type = 'ASC';
        }
        $sql = "SELECT 
            c.id,c.status,c.sort,c.parent_id,cl.name,cl.lang_code
        FROM categories c
        LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
        WHERE 1=1";



        if (!empty($searchName)) {
            $sql .= " AND cl.name LIKE :searchName";
        }
        if (!empty($searchId)) {
            $sql .= " AND c.id = :searchId";
        }
        if ($searchParentId !== null && $searchParentId !== '') {
            $sql .= " AND c.parent_id = :searchParentId";
        }
        if (!empty($searchSort)) {
            $sql .= " AND c.sort = :searchSort";
        }

        // ترتيب البيانات
        $sql .= " ORDER BY c.sort {$type}, c.id {$type}";

        // إضافة Limit لو فيه بيانات
        if (!is_null($aStart) && !is_null($aLimit)) {
            $sql .= " LIMIT :aStart, :aLimit";
        }
        try {
            $this->mDb->connect()->beginTransaction();
            $stmt = $this->mDb->connect()->prepare($sql);
            // error_log($sql);

            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);

            if (!empty($searchName)) {
                $searchName = "%{$searchName}%";
                $stmt->bindParam(':searchName', $searchName, PDO::PARAM_STR);
            }
            if (!empty($searchId)) {
                $stmt->bindParam(':searchId', $searchId, PDO::PARAM_INT);
            }
            if ($searchParentId !== null && $searchParentId !== '') {
                $stmt->bindParam(':searchParentId', $searchParentId, PDO::PARAM_INT);
            }
            if (!empty($searchSort)) {
                $stmt->bindParam(':searchSort', $searchSort, PDO::PARAM_INT);
            }

            if (!is_null($aStart) && !is_null($aLimit)) {
                $stmt->bindParam(':aStart', $aStart, PDO::PARAM_INT);
                $stmt->bindParam(':aLimit', $aLimit, PDO::PARAM_INT);
            }

            $stmt->execute();
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $this->mDb->connect()->commit();
            return $results;
        } catch (PDOException $e) {
            $this->mDb->connect()->rollBack();
            error_log("PDOException in getAllPages: " . $e->getMessage());
            return array();
        }
    }

    function getCategoryWithChildren($id, $lang_code = 'ar')
    {
        try {
            $db = $this->mDb->connect();

            $id = intval($id);
            if ($id <= 0) {
                return ['status' => 400, 'message' => 'Invalid category ID'];
            }

            // جلب بيانات القسم الحالي
            $sqlParent = "
            SELECT 
                c.id,
                c.status,
                c.image,
                c.parent_id,
                cl.name,
                cl.description
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE c.id = :id
            LIMIT 1
        ";
            $stmtParent = $db->prepare($sqlParent);
            $stmtParent->execute([
                ':id' => $id,
                ':lang_code' => $lang_code
            ]);

            $parent = $stmtParent->fetch(PDO::FETCH_ASSOC);

            if (!$parent) {
                return ['status' => 404, 'message' => 'Category not found'];
            }

            // جلب الأقسام الفرعية لهذا القسم
            $sqlChildren = "
            SELECT 
                c.id,
                c.status,
                c.image,
                c.parent_id,
                cl.name,
                cl.description
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE c.parent_id = :id
            ORDER BY c.id ASC
        ";
            $stmtChildren = $db->prepare($sqlChildren);
            $stmtChildren->execute([
                ':id' => $id,
                ':lang_code' => $lang_code
            ]);
            $children = $stmtChildren->fetchAll(PDO::FETCH_ASSOC);

            // دمج النتائج
            return [
                'status' => 200,
                'data' => [
                    'category' => $parent,
                    'children' => $children
                ]
            ];
        } catch (PDOException $e) {
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }

    function getOneCategory($id, $lang_code = 'ar')
    {
        try {
            $id = intval($id);
            if ($id <= 0) {
                return ['status' => 400, 'message' => 'Invalid category ID'];
            }

            $db = $this->mDb->connect();

            $sql = "
            SELECT 
                c.id,
                c.status,
                c.image,
                c.parent_id,
                cl.name,
                cl.description
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE c.id = :id
            LIMIT 1
        ";

            $stmt = $db->prepare($sql);
            $stmt->execute([
                ':id' => $id,
                ':lang_code' => $lang_code
            ]);

            $category = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$category) {
                return ['status' => 404, 'message' => 'Category not found'];
            }

            $category['path'] = $this->buildCategoryPathFromCategoryId($id, $lang_code);

            return [
                'status' => 200,
                'data' => $category
            ];
        } catch (PDOException $e) {
            return [
                'status' => 500,
                'message' => 'Database Error: ' . $e->getMessage()
            ];
        }
    }

    function addEditCategory($data)
    {
        try {
            $db = $this->mDb->connect();
            $db->beginTransaction();

            $id = isset($data['id']) && intval($data['id']) > 0 ? intval($data['id']) : 0;
            $status = isset($data['status']) ? $data['status'] : 'active';
            $image = isset($data['image']) ? $data['image'] : null;
            $parent_id = isset($data['parent_id']) ? intval($data['parent_id']) : 0;
            $sort = isset($data['sort']) ? intval($data['sort']) : 0; // ← قراءة قيمة sort

            // تأكد إن parent_id موجود فعلاً
            if ($parent_id > 0) {
                $checkParent = $db->prepare("SELECT COUNT(*) FROM categories WHERE id = ?");
                $checkParent->execute([$parent_id]);
                if ($checkParent->fetchColumn() == 0) {
                    return ['status' => 400, 'message' => 'Invalid parent_id (no such parent category)'];
                }
            }

            if ($id > 0) {
                // تعديل قسم
                $sql = "UPDATE categories 
                    SET status = :status, image = :image, parent_id = :parent_id, sort = :sort 
                    WHERE id = :id";
                $stmt = $db->prepare($sql);
                $stmt->execute([
                    ':status' => $status,
                    ':image' => $image,
                    ':parent_id' => $parent_id,
                    ':sort' => $sort,
                    ':id' => $id
                ]);
            } else {
                // إضافة قسم جديد
                $sql = "INSERT INTO categories (status, image, parent_id, sort, date_added)
                    VALUES (:status, :image, :parent_id, :sort, NOW())";
                $stmt = $db->prepare($sql);
                $stmt->execute([
                    ':status' => $status,
                    ':image' => $image,
                    ':parent_id' => $parent_id,
                    ':sort' => $sort
                ]);
                $id = $db->lastInsertId();
            }

            // حذف الترجمات القديمة
            $deleteLangs = $db->prepare("DELETE FROM categories_langs WHERE id_category = :id_category");
            $deleteLangs->execute([':id_category' => $id]);

            // التعامل مع اللغات المرسلة
            if (isset($data['langs'])) {
                foreach ($data['langs'] as $langItem) {
                    $lang_code = $langItem['lang_code'];
                    $name = $langItem['name'];
                    $desc = $langItem['description'];

                    $insertLang = $db->prepare("INSERT INTO categories_langs (id_category, lang_code, name, description)
                                            VALUES (:id_category, :lang_code, :name, :desc)");
                    $insertLang->execute([
                        ':id_category' => $id,
                        ':lang_code' => $lang_code,
                        ':name' => $name,
                        ':desc' => $desc
                    ]);
                }
            }

            $db->commit();

            // بناء المسار لكل لغة
            $path_ar = $this->buildCategoryPathFromCategoryId($id, 'ar');
            $path_en = $this->buildCategoryPathFromCategoryId($id, 'en');

            return [
                'status' => 200,
                'message' => $id > 0 ? 'Category updated successfully' : 'Category added successfully',
                'category_id' => $id,
                'path_ar' => $path_ar,
                'path_en' => $path_en
            ];
        } catch (PDOException $e) {
            if ($db->inTransaction()) $db->rollBack();
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }

    function buildCategoryPathFromCategoryId($category_id, $lang_code = 'ar')
    {
        $db = $this->mDb->connect();
        $path = [];

        while ($category_id != 0) {
            $stmt = $db->prepare("
            SELECT c.id, c.parent_id, cl.name
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE c.id = :id
            LIMIT 1
        ");
            $stmt->execute([
                ':id' => $category_id,
                ':lang_code' => $lang_code
            ]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$row || !$row['name']) break;

            $path[] = $row['name'];
            $category_id = $row['parent_id'];
        }

        return implode(' > ', array_reverse($path));
    }

    //     function getCategoryTree($lang_code = 'ar')
    // {
    //     try {
    //         $sql = "
    //             SELECT 
    //                 c.id,
    //                 c.parent_id,
    //                 c.status,
    //                 c.image,
    //                 cl.name,
    //                 cl.description
    //             FROM categories c
    //             LEFT JOIN categories_langs cl 
    //                 ON cl.id_cat = c.id AND cl.lang_code = :lang_code
    //             ORDER BY c.parent_id ASC, c.id ASC
    //         ";

    //         $stmt = $this->mDb->connect()->prepare($sql);
    //         $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);
    //         $stmt->execute();
    //         $categories = $stmt->fetchAll(PDO::FETCH_ASSOC);

    //         $tree = [];
    //         $map = [];

    //         foreach ($categories as &$cat) {
    //             $cat['children'] = [];
    //             $map[$cat['id']] = &$cat;
    //         }

    //         foreach ($categories as &$cat) {
    //             if ($cat['parent_id'] == 0) {
    //                 $tree[] = &$cat;
    //             } else {
    //                 if (isset($map[$cat['parent_id']])) {
    //                     $map[$cat['parent_id']]['children'][] = &$cat;
    //                 }
    //             }
    //         }

    //         return $tree;

    //     } catch (PDOException $e) {
    //         return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
    //     }
    // }




    public function getCategoriesCount($searchName = '', $searchId = '', $searchParentId = '', $searchSort = '', $lang_code = 'ar'){
        $sql = "SELECT COUNT(DISTINCT c.id)
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE 1=1";

        $hasFilters = false;

        if (!empty($searchName)) {
            $sql .= " AND cl.name LIKE :searchName";
            $hasFilters = true;
        }
        if (!empty($searchId)) {
            $sql .= " AND c.id = :searchId";
            $hasFilters = true;
        }
        if ($searchParentId !== null && $searchParentId !== '') {
            $sql .= " AND c.parent_id = :searchParentId";
            $hasFilters = true;
        }
        if (!empty($searchSort)) {
            $sql .= " AND c.sort = :searchSort";
            $hasFilters = true;
        }

        try {
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);

            if (!empty($searchName)) {
                $searchNameParam = "%{$searchName}%";
                $stmt->bindParam(':searchName', $searchNameParam, PDO::PARAM_STR);
            }
            if (!empty($searchId)) {
                $stmt->bindParam(':searchId', $searchId, PDO::PARAM_INT);
            }
            if ($searchParentId !== null && $searchParentId !== '') {
                $stmt->bindParam(':searchParentId', $searchParentId, PDO::PARAM_INT);
            }
            if (!empty($searchSort)) {
                $stmt->bindParam(':searchSort', $searchSort, PDO::PARAM_INT);
            }

            $stmt->execute();
            $count = $stmt->fetchColumn();

            return [
                'status' => 200,
                'count' => intval($count),
                'filtered' => $hasFilters
            ];
        } catch (PDOException $e) {
            return [
                'status' => 500,
                'message' => 'Database Error: ' . $e->getMessage()
            ];
        }
    }
}
