<?php

$banksBranchesObj = new bankBranches();

class bankBranches
{

	var $mDb;
	var $mConfig;
	var $mlang;
    

	function __construct()
	{
		global $Config;
		global $Lang;
		$this->mDb = new iplus();
		$this->mConfig = $Config;
		$this->mlang = $Lang;
     
	}

	function convert_object_to_array($data)
	{

		if (is_object($data)) {
			$data = get_object_vars($data);
		}

		if (is_array($data)) {
			return array_map(__METHOD__, $data);
		} else {
			return $data;
		}
	}
   

    function getAllBranches($aStart, $aLimit, $sort, $type, $searchName, $searchBankId) {
        $lang_code = LANGUAGE;
        
        // Validate if the searchBankId exists in the branches table
        if ($searchBankId) {
            $validationSql = "SELECT COUNT(*) AS count FROM branches WHERE bank_id = :searchBankId AND is_deleted = 0";
            try {
                $validationStmt = $this->mDb->connect()->prepare($validationSql);
                $validationStmt->bindParam(':searchBankId', $searchBankId, PDO::PARAM_INT);
                $validationStmt->execute();
                $validationResult = $validationStmt->fetch(PDO::FETCH_ASSOC);
    
                if ($validationResult['count'] == 0) {
                    return ['error' => "Bank ID {$searchBankId} does not exist in the branches table."];
                }
            } catch (PDOException $e) {
                error_log("PDOException in validating searchBankId: " . $e->getMessage());
                return ['error' => "An error occurred while validating the Bank ID."];
            }
        }
    
        // Main query
        $sql = "SELECT b.*, bl.name, bl.address
                FROM branches b
                LEFT JOIN branches_langs bl ON b.id = bl.branch_id AND bl.lang_code = :lang_code
                WHERE b.is_deleted = 0 ";
    
        if ($searchName) {
            $sql .= " AND bl.name LIKE :searchName";
        }
        if ($searchBankId) {
            $sql .= " AND b.bank_id = :searchBankId";
        }
        if ($sort == 'name') {
            $sql .= " ORDER BY bl.name {$type}";
        } else {
            $sql .= " ORDER BY b.{$sort} {$type}";
        }
        if ($aLimit) {
            $sql .= " LIMIT :aStart, :aLimit";
        }
        // echo ($sql);die();
        try {
            $this->mDb->connect()->beginTransaction();
    
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);
    
            if ($searchName) {
                $stmt->bindParam(':searchName', $searchName, PDO::PARAM_STR);
            }
            if ($searchBankId) {
                $stmt->bindParam(':searchBankId', $searchBankId, PDO::PARAM_INT);
            }
    
            if ($aLimit) {
                $stmt->bindParam(':aStart', $aStart, PDO::PARAM_INT);
                $stmt->bindParam(':aLimit', $aLimit, PDO::PARAM_INT);
            }
    
            $stmt->execute();
            $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
            $this->mDb->connect()->commit();
            return $results;
    
        } catch (PDOException $e) {
            $this->mDb->connect()->rollBack();
            error_log("PDOException in getAllPages: " . $e->getMessage());
            return array();
        }
    }
    
    function getSearchBranchesCount($searchName, $searchBankId) {
        $sql = "SELECT COUNT(b.`id`) as result_count 
                FROM branches b
                LEFT JOIN branches_langs bl ON b.id = bl.branch_id AND bl.lang_code = :lang_code
                WHERE b.is_deleted = 0 ";
    
        if (!empty($searchName)) {
            $sql .= " AND bl.name LIKE :searchName";
        }
        if (!empty($searchBankId)) {
            $sql .= " AND bl.bank_id LIKE :searchBankId";
        }
    
        try {
            $this->mDb->connect()->beginTransaction();
    
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':lang_code', $lang_code, PDO::PARAM_STR);
    
            if (!empty($searchName)) {
                $stmt->bindParam(':searchName', $searchName, PDO::PARAM_STR);
            }
            if (!empty($searchBankId)) {
                $stmt->bindParam(':searchBankId', $searchBankId, PDO::PARAM_INT);
            }
    
            $stmt->execute();
            $resultCount = $stmt->fetchColumn();
    
            $this->mDb->connect()->commit();
            return $resultCount;
    
        } catch (PDOException $e) {
            $this->mDb->connect()->rollBack();
            error_log("PDOException in getSearchBanksCount: " . $e->getMessage());
            return 0;
        }
    }

    function getOneBranch($id) {
        try {
            $this->mDb->beginTransaction();
    
            $sql = "SELECT b.*
                    FROM `branches` b WHERE b.`id` = :id";
    
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':id', $id, PDO::PARAM_INT);
    
            if ($stmt->execute()) {
                $result = $stmt->fetch(PDO::FETCH_ASSOC);
                if (!$result) {
                    throw new Exception("branch with ID $id not found.");
                }
            } else {
                throw new Exception("Error executing the main SQL statement: " . implode(", ", $stmt->errorInfo()));
            }
    
            $sqlLangs = "SELECT bl.`lang_code`, bl.`name`, bl.`address`
                         FROM `branches_langs` bl WHERE bl.`branch_id` = :id";
            $stmtLangs = $this->mDb->connect()->prepare($sqlLangs);
            $stmtLangs->bindParam(':id', $id, PDO::PARAM_INT);
    
            if ($stmtLangs->execute()) {
                $result['langs'] = $stmtLangs->fetchAll(PDO::FETCH_ASSOC);
            } else {
                throw new Exception("Error executing the languages SQL statement: " . implode(", ", $stmtLangs->errorInfo()));
            }
    
            // Commit the transaction after both queries are successful
            $this->mDb->commit();
    
            return $result;
        } catch (Exception $e) {
            // Rollback in case of any error
            $this->mDb->rollBack();
            return ["status" => 500, "message" => "An error occurred: " . $e->getMessage()];
        }
    }
    function getBankBranches($bankId) {
        try {
            $this->mDb->beginTransaction();
    
            // Query to get all branches for the given bank_id
            $sql = "SELECT b.*
                    FROM branches b 
                    WHERE b.bank_id = :bank_id AND b.is_deleted = 0";
    
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->bindParam(':bank_id', $bankId, PDO::PARAM_INT);
    
            if ($stmt->execute()) {
                $branches = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
                if (!$branches) {
                    throw new Exception("No branches found for bank ID $bankId.");
                }
            } else {
                throw new Exception("Error executing branches SQL: " . implode(", ", $stmt->errorInfo()));
            }
    
            $result = ['branches' => []];
    
            foreach ($branches as $branch) {
                $branchId = $branch['id'];
    
                $sqlLangs = "SELECT bl.lang_code, bl.name, bl.address
                             FROM branches_langs bl 
                             WHERE bl.branch_id = :branch_id";
    
                $stmtLangs = $this->mDb->connect()->prepare($sqlLangs);
                $stmtLangs->bindParam(':branch_id', $branchId, PDO::PARAM_INT);
    
                if ($stmtLangs->execute()) {
                    $branchLangs = $stmtLangs->fetchAll(PDO::FETCH_ASSOC);
    
                    // Add branch data along with its language details directly
                    $branch['langs'] = $branchLangs;
                    $result['branches'][] = $branch;
                } else {
                    throw new Exception("Error executing branch_langs SQL for branch ID $branchId: " . implode(", ", $stmtLangs->errorInfo()));
                }
            }
    
            // Commit the transaction
            $this->mDb->commit();
    
            return $result ;
    
        } catch (Exception $e) {
            // Rollback in case of any error
            $this->mDb->rollBack();
            return ["status" => 500, "message" => "An error occurred: " . $e->getMessage()];
        }
    }
    
    function addEditBankBranch($request) {
        // Validate 'added_by' field
        if (isset($request['added_by']) && !is_int($request['added_by']) && !ctype_digit($request['added_by'])) {
            return ["status" => 400, "message" => "'added_by' must be an integer"];
        }
    
        unset($request['action']);
        $dataLangObj = json_decode($request['langs']);
        $newDataLangObj = $this->convert_object_to_array($dataLangObj);
        $dateTime = date('Y-m-d H:i:s');
    
        // Check if bank exists
        $sqlCheckBank = "SELECT COUNT(*) FROM banks WHERE id = :bank_id";
        $stmtCheckBank = $this->mDb->connect()->prepare($sqlCheckBank);
        $stmtCheckBank->bindParam(':bank_id', $request['bank_id'], PDO::PARAM_INT);
        $stmtCheckBank->execute();
        $bankExists = $stmtCheckBank->fetchColumn();
    
        if (!$bankExists) {
            return ["status" => 404, "message" => "The bank does not exist"];
        }
    
        if (!($request['id'])) {
            // Insert operation
            $sql = "INSERT INTO branches SET added_by = :added_by, date_added = :dateTime";
            $sql .= ", bank_id = :bank_id";
            $sql .= ", latitude = :latitude";
            $sql .= ", longitude = :longitude";
            $sql .= ", is_main_branch = :is_main_branch";
            $sql .= ", working_days = :working_days";
            $sql .= ", working_from = :working_from";
            $sql .= ", working_to = :working_to";
            $sql .= ", status = :status";
            $sql .= ", sort = :sort";
    
            try {
                $this->mDb->beginTransaction();
                $stmt = $this->mDb->connect()->prepare($sql);
    
                $stmt->bindParam(':added_by', $request['added_by'], PDO::PARAM_INT);
                $stmt->bindParam(':dateTime', $dateTime);
                $stmt->bindParam(':bank_id', $request['bank_id'], PDO::PARAM_INT);
                $stmt->bindParam(':latitude', $request['latitude']);
                $stmt->bindParam(':longitude', $request['longitude']);
                $stmt->bindParam(':is_main_branch', $request['is_main_branch'], PDO::PARAM_INT);
                $stmt->bindParam(':working_days', $request['working_days']);
                $stmt->bindParam(':working_from', $request['working_from']);
                $stmt->bindParam(':working_to', $request['working_to']);
                $stmt->bindParam(':status', $request['status'], PDO::PARAM_INT);
                $stmt->bindParam(':sort', $request['sort'], PDO::PARAM_INT);
    
                $stmt->execute();
    
                $last_id = $this->mDb->getLastInsertId();
                foreach ($newDataLangObj as $key => $value) {
                    $sqlForLang = "INSERT INTO branches_langs (branch_id, lang_code, address, name) VALUES (:branch_id, :lang_code, :address, :name)";
                    $stmtForLang = $this->mDb->connect()->prepare($sqlForLang);
                    $stmtForLang->bindParam(':branch_id', $last_id, PDO::PARAM_INT);
                    $stmtForLang->bindParam(':lang_code', $value['lang_code']);
                    $stmtForLang->bindParam(':address', $value['address'], PDO::PARAM_STR);
                    $stmtForLang->bindParam(':name', $value['name'], PDO::PARAM_STR);
                    $stmtForLang->execute();
                }
    
                $this->mDb->commit();
                return ["status" => 200, "message" => "Branch added successfully"];
            } catch (PDOException $e) {
                $this->mDb->rollBack();
                return ["status" => 500, "message" => "An error occurred"];
            }
        } else {
            // Update operation
            $sql = "UPDATE branches SET date_added = :date_added, bank_id = :bank_id, latitude = :latitude, 
                    longitude = :longitude, is_main_branch = :is_main_branch, working_days = :working_days, 
                    working_from = :working_from, working_to = :working_to, status = :status, sort = :sort 
                    WHERE id = :id";
    
            try {
                $this->mDb->beginTransaction();
                $stmt = $this->mDb->connect()->prepare($sql);
    
                $stmt->bindParam(':date_added', $request['date_added']);
                $stmt->bindParam(':bank_id', $request['bank_id'], PDO::PARAM_INT);
                $stmt->bindParam(':latitude', $request['latitude']);
                $stmt->bindParam(':longitude', $request['longitude']);
                $stmt->bindParam(':is_main_branch', $request['is_main_branch'], PDO::PARAM_INT);
                $stmt->bindParam(':working_days', $request['working_days']);
                $stmt->bindParam(':working_from', $request['working_from']);
                $stmt->bindParam(':working_to', $request['working_to']);
                $stmt->bindParam(':status', $request['status'], PDO::PARAM_INT);
                $stmt->bindParam(':sort', $request['sort'], PDO::PARAM_INT);
                $stmt->bindParam(':id', $request['id'], PDO::PARAM_INT);
    
                $stmt->execute();
    
                // Update languages for the branch
                foreach ($newDataLangObj as $key => $value) {
                    $sqlForLang = "UPDATE branches_langs SET name = :name, address = :address WHERE branch_id = :branch_id AND lang_code = :lang_code";
                    $stmtForLang = $this->mDb->connect()->prepare($sqlForLang);
                    $stmtForLang->bindParam(':name', $value['name']);
                    $stmtForLang->bindParam(':address', $value['address']);
                    $stmtForLang->bindParam(':branch_id', $request['id'], PDO::PARAM_INT);
                    $stmtForLang->bindParam(':lang_code', $value['lang_code']);
                    $stmtForLang->execute();
                }
    
                $this->mDb->commit();
                return ["status" => 200, "message" => "Branch updated successfully"];
            } catch (PDOException $e) {
                $this->mDb->rollBack();
                return ["status" => 500, "message" => "An error occurred"];
            }
        }
    }

    function deleteBranch($ids) {
        if (empty($ids) || !is_array($ids)) {
            return [
                "status" => 400,
                "message" => "Invalid input. Please provide an array of IDs."
            ];
        }
    
        try {
            $pdo = $this->mDb->connect();
            $pdo->beginTransaction();
    
            // Prepare to fetch the logos (images) of the banks to delete
            $placeholders = implode(',', array_fill(0, count($ids), '?'));
            $sqlDeleteLangs = "DELETE FROM `branches_langs` WHERE `branch_id` IN ($placeholders)";
            $stmtDeleteLangs = $this->mDb->connect()->prepare($sqlDeleteLangs);
            $stmtDeleteLangs->execute($ids);
    
            // Check if all the bank IDs exist
            $sqlCheck = "SELECT `id` FROM `branches` WHERE `id` IN ($placeholders)";
            $stmtCheck = $pdo->prepare($sqlCheck);
    
            foreach ($ids as $index => $id) {
                $stmtCheck->bindValue($index + 1, $id, PDO::PARAM_INT);
            }
    
            $stmtCheck->execute();
            $existingIds = $stmtCheck->fetchAll(PDO::FETCH_COLUMN);
    
            // Check for non-existing IDs
            $nonExistingIds = array_diff($ids, $existingIds);
            if (!empty($nonExistingIds)) {
                $nonExistingIdsString = implode(', ', $nonExistingIds);
                return [
                    "status" => 404,
                    "message" => "The following IDs do not exist in the database: $nonExistingIdsString"
                ];
            }
    
            // Perform the deletion
            $sqlDelete = "DELETE FROM `branches` WHERE `id` IN ($placeholders)";
            $stmtDelete = $pdo->prepare($sqlDelete);
    
            foreach ($ids as $index => $id) {
                $stmtDelete->bindValue($index + 1, $id, PDO::PARAM_INT);
            }
    
            if ($stmtDelete->execute()) {
                $pdo->commit();
                return [
                    "status" => 200,
                    "message" => "barnch(es) deleted successfully."
                ];
            } else {
                $pdo->rollBack();
                return [
                    "status" => 500,
                    "message" => "Failed to delete branch(es)."
                ];
            }
        } catch (PDOException $e) {
            $pdo->rollBack();
            error_log("PDOException: " . $e->getMessage());
    
            return [
                "status" => 500,
                "message" => "Error occurred while deleting bank(s): " . $e->getMessage()
            ];
        }
    }
    
    

}