<?php

$articlesObj = new articles();

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Cell\DataValidation;

class articles
{

    var $mDb;
    var $mConfig;
    var $mlang;


    function __construct()
    {
        global $Config;
        global $Lang;
        $this->mDb = new iplus();
        $this->mConfig = $Config;
        $this->mlang = $Lang;
    }

    function convert_object_to_array($data)
    {

        if (is_object($data)) {
            $data = get_object_vars($data);
        }

        if (is_array($data)) {
            return array_map(__METHOD__, $data);
        } else {
            return $data;
        }
    }
    public function styleHeaderRow($sheet, $range)
    {
        $sheet->getStyle($range)->applyFromArray([
            'font' => ['bold' => true],
            'alignment' => ['horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER],
            'fill' => [
                'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                'startColor' => ['rgb' => 'D9E1F2']
            ]
        ]);
    }
    public function styleRow($sheet, $rowIndex, $startCol = 'A', $endCol = 'E')
    {
        $range = "{$startCol}{$rowIndex}:{$endCol}{$rowIndex}";
        $sheet->getStyle($range)->getAlignment()->setVertical(\PhpOffice\PhpSpreadsheet\Style\Alignment::VERTICAL_CENTER);
        $sheet->getStyle($range)->getAlignment()->setWrapText(true);
        $sheet->getStyle($range)->getBorders()->getAllBorders()->setBorderStyle(\PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN);
    }


    public function buildCategoryFullPath($catId, $lang = 'ar')
    {
        $db = $this->mDb->connect();
        $path = [];

        // تأمين id
        $currentId = intval($catId);
        $currentId = $catId;


        while ($currentId > 0) {


            $stmt = $db->prepare("
            SELECT c.id, c.parent_id, cl.name
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = ?
            WHERE c.id = ?
        ");

            $stmt->execute([$lang, $currentId]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$row) break;

            $name = is_string($row['name']) ? $row['name'] : 'بدون اسم';
            array_unshift($path, strval($name));
            $currentId = intval($row['parent_id']);
        }

        return implode(' > ', $path); // ← تأكد إنها كده
    }



    public function getAllLeafCategoryPaths($lang = 'ar')
    {
        $db = $this->mDb->connect();

        // جلب كل الأقسام التي ليست أبًا لأي قسم آخر
        $stmt = $db->query("
        SELECT c.id
        FROM categories c
        LEFT JOIN categories c2 ON c.id = c2.parent_id
        WHERE c2.id IS NULL
    ");
        $leafIds = $stmt->fetchAll(PDO::FETCH_COLUMN);

        $paths = [];
        foreach ($leafIds as $catId) {
            $path = $this->buildCategoryFullPath($catId, $lang);
            if ($path !== '') {
                $paths[] = $path . ' {' . $catId . '}';
            }
        }

        return $paths;
    }


    public function exportArticlesExcel(array $data)
    {
        $headersMain = ['#', 'ID المقال', 'الحالة', 'الترتيب'];
        $headersLang = ['#', 'ID المقال', 'اللغة', 'العنوان', 'المحتوى'];
        $headersEdit = ['#', 'ID المقال', 'المسار'];

        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $db = $this->mDb->connect();
        $lang = $_GET['lang'] ?? 'ar'; // 🟢 اللغة المطلوبة

        // 🟩 شيت المقالات
        $sheetMain = $spreadsheet->getActiveSheet();
        $sheetMain->setTitle('المقالات');
        $sheetMain->fromArray($headersMain, null, 'A1');
        $this->styleHeaderRow($sheetMain, 'A1:D1');

        $rowIndex = 2;
        $serial = 1;

        foreach ($data as $row) {
            $articleId = $row['id'] ?? 0;
            $statusText = $row['status'] == 1 ? 'فعال (1)' : 'معطل (0)';
            $sort = $row['sort'] ?? '';

            $sheetMain->fromArray([
                $serial++,
                $articleId,
                $statusText,
                $sort
            ], null, 'A' . $rowIndex);

            $this->styleRow($sheetMain, $rowIndex, 'A', 'D');
            $rowIndex++;
        }

        foreach (range('A', 'D') as $col) {
            $sheetMain->getColumnDimension($col)->setAutoSize(true);
        }

        $statusDV = new \PhpOffice\PhpSpreadsheet\Cell\DataValidation();
        $statusDV->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
        $statusDV->setFormula1('"فعال (1),معطل (0)"');
        $statusDV->setAllowBlank(true);
        $statusDV->setShowDropDown(true);

        for ($r = 2; $r < $rowIndex; $r++) {
            $sheetMain->getCell("C{$r}")->setDataValidation(clone $statusDV);
        }

        // 🟦 شيت اللغات
        $sheetLang = $spreadsheet->createSheet();
        $sheetLang->setTitle('اللغات');
        $sheetLang->fromArray($headersLang, null, 'A1');
        $this->styleHeaderRow($sheetLang, 'A1:E1');

        $rowLang = 2;
        $serialLang = 1;

        foreach ($data as $row) {
            $articleId = $row['id'] ?? 0;
            $stmt = $db->prepare("SELECT lang_code, title, content FROM articles_langs WHERE id_article = ?");
            $stmt->execute([$articleId]);
            $langs = $stmt->fetchAll(PDO::FETCH_ASSOC);

            foreach ($langs as $langRow) {
                $sheetLang->fromArray([
                    $serialLang++,
                    $articleId,
                    $langRow['lang_code'],
                    $langRow['title'],
                    $langRow['content']
                ], null, 'A' . $rowLang);
                $this->styleRow($sheetLang, $rowLang, 'A', 'E');
                $rowLang++;
            }
        }

        foreach (range('A', 'E') as $col) {
            $sheetLang->getColumnDimension($col)->setAutoSize(true);
        }

        // 🟨 شيت التعديل على المقالات
        $sheetEdit = $spreadsheet->createSheet();
        $sheetEdit->setTitle('التعديل على المقالات');
        $sheetEdit->fromArray($headersEdit, null, 'A1');
        $this->styleHeaderRow($sheetEdit, 'A1:C1');

        $serialEdit = 1;
        $rowEdit = 2;
        $colors = ['#f0f8ff', '#f9f9f9', '#fffbe6'];
        $currentColorIndex = 0;

        foreach ($data as $row) {
            $articleId = $row['id'] ?? 0;

            $stmt = $db->prepare("SELECT id_category FROM articles_categories WHERE id_article = ?");
            $stmt->execute([$articleId]);
            $catIds = $stmt->fetchAll(PDO::FETCH_COLUMN);

            $startRow = $rowEdit;
            $rowsToColor = 0;

            if (!empty($catIds)) {
                foreach ($catIds as $catIdRaw) {
                    $catId = is_array($catIdRaw) ? intval($catIdRaw['id_category'] ?? 0) : intval($catIdRaw);
                    if ($catId <= 0) continue;

                    $path = $this->buildCategoryFullPath($catId, $lang);
                    $pathWithId = $path !== '' ? $path . ' {' . $catId . '}' : '{' . $catId . '}';

                    $sheetEdit->fromArray([
                        $serialEdit++,
                        $articleId,
                        $pathWithId
                    ], null, 'A' . $rowEdit);
                    $this->styleRow($sheetEdit, $rowEdit, 'A', 'C');
                    $rowsToColor++;
                    $rowEdit++;
                }
            } else {
                $sheetEdit->fromArray([
                    $serialEdit++,
                    $articleId,
                    'لا يوجد أقسام'
                ], null, 'A' . $rowEdit);
                $this->styleRow($sheetEdit, $rowEdit, 'A', 'C');
                $rowsToColor++;
                $rowEdit++;
            }

            if ($rowsToColor > 0) {
                $fillColor = $colors[$currentColorIndex % count($colors)];
                for ($i = $startRow; $i < $startRow + $rowsToColor; $i++) {
                    $sheetEdit->getStyle("A{$i}:C{$i}")->getFill()->setFillType(\PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID);
                    $sheetEdit->getStyle("A{$i}:C{$i}")->getFill()->getStartColor()->setRGB(str_replace('#', '', $fillColor));
                }
                $currentColorIndex++;
            }
        }

        foreach (range('A', 'C') as $col) {
            $sheetEdit->getColumnDimension($col)->setAutoSize(true);
        }

        // 🔒 شيت الأقسام النهائية فقط (مرجعي للدروب ليست)
        $leafPaths = $this->getAllLeafCategoryPaths($lang);
        $refSheet = $spreadsheet->createSheet();
        $refSheet->setTitle('categories_reference');
        $refSheet->setSheetState(\PhpOffice\PhpSpreadsheet\Worksheet\Worksheet::SHEETSTATE_HIDDEN);
        foreach ($leafPaths as $i => $path) {
            $refSheet->setCellValue('A' . ($i + 1), $path);
        }

        $catDV = new \PhpOffice\PhpSpreadsheet\Cell\DataValidation();
        $catDV->setType(\PhpOffice\PhpSpreadsheet\Cell\DataValidation::TYPE_LIST);
        $catDV->setFormula1('=categories_reference!$A$1:$A$' . count($leafPaths));
        $catDV->setAllowBlank(true);
        $catDV->setShowDropDown(true);

        for ($r = 2; $r < $rowEdit; $r++) {
            $sheetEdit->getCell("C{$r}")->setDataValidation(clone $catDV);
        }

        // 🟢 حفظ الملف
        $spreadsheet->setActiveSheetIndex(0);
        $fileName = 'articles_export_' . date('Ymd_His') . '.xlsx';
        $filePath = __DIR__ . '../../../exports/' . $fileName;

        if (!is_dir(dirname($filePath))) {
            mkdir(dirname($filePath), 0777, true);
        }

        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        $writer->save($filePath);

        return [
            'status' => 200,
            'message' => 'تم إنشاء ملف الإكسل بنجاح',
            'file' => 'exports/' . $fileName
        ];
    }


    function importArticlesExcel($file)
    {
        try {
            if (!isset($file['tmp_name']) || !file_exists($file['tmp_name'])) {
                return ['status' => 400, 'message' => 'لم يتم رفع الملف بشكل صحيح'];
            }

            $allowedExtensions = ['xls', 'xlsx'];
            $fileExtension = pathinfo($file['name'], PATHINFO_EXTENSION);

            if (!in_array(strtolower($fileExtension), $allowedExtensions)) {
                return ['status' => 400, 'message' => 'يجب رفع ملف Excel بصيغة xls أو xlsx فقط'];
            }

            $db = $this->mDb->connect();
            $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($file['tmp_name']);

            // 🟢 قراءة الشيتات
            $mainSheet = $spreadsheet->getSheetByName('المقالات');
            $langSheet = $spreadsheet->getSheetByName('اللغات');
            $editSheet = $spreadsheet->getSheetByName('التعديل على المقالات');

            if (!$mainSheet || !$langSheet || !$editSheet) {
                return ['status' => 400, 'message' => 'الملف لا يحتوي على الشيتات المطلوبة'];
            }

            $mainData = $mainSheet->toArray();
            $langData = $langSheet->toArray();
            $editData = $editSheet->toArray();

            $updated = 0;
            $inserted = 0;
            $deletedArticles = 0;
            $deletedLangs = 0;
            $categoriesUpdated = 0;
            $categoriesDeleted = 0;
            $categoriesAdded = 0;
            $langsUpdated = 0;
            $langsInserted = 0;

            // 🔍 أولاً: تحديد النطاق من أول مقال وآخر مقال في الملف لفهم النطاق المطلوب
            $allArticleIdsInFile = [];
            for ($i = 1; $i < count($mainData); $i++) {
                $row = $mainData[$i];
                if (empty($row[1]) && empty($row[2]) && empty($row[3])) {
                    continue;
                }
                $articleId = intval($row[1]);
                if ($articleId > 0) {
                    $allArticleIdsInFile[] = $articleId;
                }
            }

            // إزالة التكرارات وترتيب للحصول على النطاق
            $allArticleIdsInFile = array_unique($allArticleIdsInFile);
            sort($allArticleIdsInFile);

            if (empty($allArticleIdsInFile)) {
                return ['status' => 400, 'message' => 'لا يوجد مقالات صالحة في الملف'];
            }

            // تحديد النطاق: من أصغر ID إلى أكبر ID في الملف
            $minId = min($allArticleIdsInFile);
            $maxId = max($allArticleIdsInFile);

            // 🟢 جلب جميع المقالات الحالية في هذا النطاق من قاعدة البيانات
            $stmt = $db->prepare("SELECT id FROM articles WHERE id BETWEEN ? AND ?");
            $stmt->execute([$minId, $maxId]);
            $originalArticleIds = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'id');

            // 🟢 جلب جميع لغات المقالات الحالية في هذا النطاق
            $stmt = $db->prepare("SELECT id_article, lang_code FROM articles_langs WHERE id_article BETWEEN ? AND ?");
            $stmt->execute([$minId, $maxId]);
            $originalLangs = $stmt->fetchAll(PDO::FETCH_ASSOC);

            $processedArticleIds = [];
            $excelLangs = [];
            $articleSections = []; // ✅ لحفظ الأقسام المرتبطة بكل مقال

            // 🟡 معالجة المقالات الرئيسية من شيت "المقالات"
            for ($i = 1; $i < count($mainData); $i++) {
                $row = $mainData[$i];

                // تخطي الصفوف الفارغة
                if (empty($row[1]) && empty($row[2]) && empty($row[3])) {
                    continue;
                }

                $articleId = intval($row[1]);
                $status = (strpos($row[2], '1') !== false) ? 1 : 0;
                $sort = intval($row[3]);

                if ($articleId > 0) {
                    $processedArticleIds[] = $articleId;

                    // فحص إذا كان المقال موجود
                    $stmt = $db->prepare("SELECT sort, status FROM articles WHERE id = ?");
                    $stmt->execute([$articleId]);
                    $existing = $stmt->fetch(PDO::FETCH_ASSOC);

                    if ($existing) {
                        // تحديث المقال إذا تغيرت البيانات
                        if (intval($existing['sort']) != $sort || intval($existing['status']) != $status) {
                            $stmtUpdate = $db->prepare("UPDATE articles SET sort = ?, status = ? WHERE id = ?");
                            $stmtUpdate->execute([$sort, $status, $articleId]);
                            $updated++;
                        }
                    } else {
                        // إضافة مقال جديد بـ ID محدد (إذا كان ممكناً)
                        try {
                            $stmtInsert = $db->prepare("INSERT INTO articles (id, sort, status, date_added) VALUES (?, ?, ?, NOW())");
                            $stmtInsert->execute([$articleId, $sort, $status]);
                            $inserted++;
                        } catch (Exception $e) {
                            // إذا فشل الإدراج بـ ID محدد، أدرج بدون ID
                            $stmtInsert = $db->prepare("INSERT INTO articles (sort, status, date_added) VALUES (?, ?, NOW())");
                            $stmtInsert->execute([$sort, $status]);
                            $newId = $db->lastInsertId();
                            $inserted++;

                            // استبدال المعرف القديم بالجديد في المتغيرات
                            $key = array_search($articleId, $processedArticleIds);
                            if ($key !== false) {
                                $processedArticleIds[$key] = $newId;
                            }
                            $articleId = $newId;
                        }
                    }
                }
            }

            // 🟦 معالجة الأقسام من شيت "التعديل على المقالات"
            for ($i = 1; $i < count($editData); $i++) {
                $row = $editData[$i];

                if (empty($row[1]) && empty($row[2])) {
                    continue;
                }

                $articleId = intval($row[1]);
                $categoryPath = trim($row[2] ?? '');

                if ($articleId > 0 && !empty($categoryPath) && $categoryPath !== 'لا يوجد أقسام') {
                    // استخراج id_category من النص
                    preg_match('/\{(\d+)\}$/', $categoryPath, $match);
                    $categoryId = isset($match[1]) ? intval($match[1]) : 0;

                    if ($categoryId > 0) {
                        if (!isset($articleSections[$articleId])) {
                            $articleSections[$articleId] = [];
                        }
                        if (!in_array($categoryId, $articleSections[$articleId])) {
                            $articleSections[$articleId][] = $categoryId;
                        }
                    }
                } else if ($articleId > 0) {
                    // إذا كان المقال موجود لكن بدون أقسام أو "لا يوجد أقسام"
                    if (!isset($articleSections[$articleId])) {
                        $articleSections[$articleId] = [];
                    }
                }
            }

            // 🟨 تحديث الأقسام المرتبطة (لجميع المقالات المعالجة)
            foreach ($processedArticleIds as $articleId) {
                // جلب الأقسام الحالية للمقال
                $stmt = $db->prepare("SELECT id_category FROM articles_categories WHERE id_article = ?");
                $stmt->execute([$articleId]);
                $currentCategories = array_column($stmt->fetchAll(PDO::FETCH_ASSOC), 'id_category');

                $newCategories = isset($articleSections[$articleId]) ? $articleSections[$articleId] : [];

                // مقارنة الأقسام الحالية مع الجديدة
                $currentCategoriesSorted = $currentCategories;
                $newCategoriesSorted = $newCategories;
                sort($currentCategoriesSorted);
                sort($newCategoriesSorted);

                if ($currentCategoriesSorted !== $newCategoriesSorted) {
                    // حساب الأقسام المحذوفة والمضافة
                    $deletedCats = array_diff($currentCategories, $newCategories);
                    $addedCats = array_diff($newCategories, $currentCategories);

                    if (!empty($deletedCats)) {
                        $categoriesDeleted += count($deletedCats);
                    }
                    if (!empty($addedCats)) {
                        $categoriesAdded += count($addedCats);
                    }

                    // حذف الأقسام القديمة
                    $db->prepare("DELETE FROM articles_categories WHERE id_article = ?")->execute([$articleId]);

                    // إضافة الأقسام الجديدة
                    foreach ($newCategories as $categoryId) {
                        $stmtInsertCat = $db->prepare("INSERT INTO articles_categories (id_article, id_category) VALUES (?, ?)");
                        $stmtInsertCat->execute([$articleId, $categoryId]);
                    }
                    $categoriesUpdated++;
                }
            }

            // 🟩 معالجة اللغات من شيت "اللغات"
            for ($i = 1; $i < count($langData); $i++) {
                $row = $langData[$i];

                if (empty($row[1]) && empty($row[2])) {
                    continue;
                }

                $articleId = intval($row[1]);
                $langCode = trim($row[2] ?? '');
                $title = trim($row[3] ?? '');
                $content = trim($row[4] ?? '');

                if (!$articleId || !$langCode) continue;

                $excelLangs[] = ['id_article' => $articleId, 'lang_code' => $langCode];

                // فحص إذا كانت الترجمة موجودة
                $stmtCheckLang = $db->prepare("SELECT title, content FROM articles_langs WHERE id_article = ? AND lang_code = ?");
                $stmtCheckLang->execute([$articleId, $langCode]);
                $existingLang = $stmtCheckLang->fetch(PDO::FETCH_ASSOC);

                if ($existingLang) {
                    // تحديث الترجمة إذا تغيرت
                    if ($existingLang['title'] !== $title || $existingLang['content'] !== $content) {
                        $stmtLang = $db->prepare("UPDATE articles_langs SET title = ?, content = ? WHERE id_article = ? AND lang_code = ?");
                        $stmtLang->execute([$title, $content, $articleId, $langCode]);
                        $langsUpdated++;
                    }
                } else {
                    // إضافة ترجمة جديدة
                    $stmtLangInsert = $db->prepare("INSERT INTO articles_langs (id_article, lang_code, title, content) VALUES (?, ?, ?, ?)");
                    $stmtLangInsert->execute([$articleId, $langCode, $title, $content]);
                    $langsInserted++;
                }
            }

            // 🔴 حذف المقالات الناقصة من الملف (بناء على النطاق المحدد)
            $processedArticleIds = array_unique($processedArticleIds);
            $articlesToDelete = array_diff($originalArticleIds, $processedArticleIds);

            if (!empty($articlesToDelete)) {
                foreach ($articlesToDelete as $articleId) {
                    // حذف الأقسام المرتبطة
                    $db->prepare("DELETE FROM articles_categories WHERE id_article = ?")->execute([$articleId]);
                    // حذف الترجمات
                    $db->prepare("DELETE FROM articles_langs WHERE id_article = ?")->execute([$articleId]);
                    // حذف المقال نفسه
                    $db->prepare("DELETE FROM articles WHERE id = ?")->execute([$articleId]);
                    $deletedArticles++;
                }
            }

            // 🔴 حذف الترجمات الناقصة من الملف (للمقالات في النطاق المحدد فقط)
            if (!empty($originalLangs)) {
                foreach ($originalLangs as $originalLang) {
                    // فحص إذا كان المقال ما زال موجوداً (لم يُحذف)
                    if (in_array($originalLang['id_article'], $processedArticleIds)) {
                        $found = false;
                        foreach ($excelLangs as $excelLang) {
                            if (
                                $originalLang['id_article'] == $excelLang['id_article'] &&
                                $originalLang['lang_code'] == $excelLang['lang_code']
                            ) {
                                $found = true;
                                break;
                            }
                        }
                        // إذا لم تجد الترجمة في الإكسل، احذفها
                        if (!$found) {
                            $db->prepare("DELETE FROM articles_langs WHERE id_article = ? AND lang_code = ?")->execute([
                                $originalLang['id_article'],
                                $originalLang['lang_code']
                            ]);
                            $deletedLangs++;
                        }
                    }
                }
            }

            // إعداد رسالة النتيجة
            $messages = [];
            if ($inserted > 0) $messages[] = "تم إضافة {$inserted} مقال جديد";
            if ($updated > 0) $messages[] = "تم تحديث {$updated} مقال";
            if ($langsUpdated > 0) $messages[] = "تم تحديث {$langsUpdated} ترجمة";
            if ($langsInserted > 0) $messages[] = "تم إضافة {$langsInserted} ترجمة جديدة";
            if ($categoriesUpdated > 0) $messages[] = "تم تحديث أقسام {$categoriesUpdated} مقال";
            if ($categoriesAdded > 0) $messages[] = "تم إضافة {$categoriesAdded} قسم جديد";
            if ($categoriesDeleted > 0) $messages[] = "تم حذف {$categoriesDeleted} قسم";
            if ($deletedArticles > 0) $messages[] = "تم حذف {$deletedArticles} مقال";
            if ($deletedLangs > 0) $messages[] = "تم حذف {$deletedLangs} ترجمة";

            if (empty($messages)) {
                return [
                    'status' => 200,
                    'message' => 'لا يوجد أي تعديل أو إضافة في الملف'
                ];
            }

            return [
                'status' => 200,
                'message' => implode('، ', $messages) . '.'
            ];
        } catch (Exception $e) {
            return ['status' => 500, 'message' => 'حدث خطأ أثناء قراءة الملف: ' . $e->getMessage()];
        }
    }







    function getAllArticles($aStart = 0, $aLimit = 10, $sort = 'id', $type = 'DESC',  $lang_code = 'ar')
    {
        try {
            $db = $this->mDb->connect();

            $sql = "SELECT 
                    a.id,

                    a.image,
                    a.sort,
                    a.date_added,
                    al.title,
                    al.content
                FROM articles a
                LEFT JOIN articles_langs al ON al.id_article = a.id AND al.lang_code = :lang_code
                WHERE 1
                ";

            $params = [
                ':lang_code' => $lang_code
            ];

            // الترتيب والفرز
            $allowedSortFields = ['id', 'sort', 'date_added'];
            $sort = in_array($sort, $allowedSortFields) ? $sort : 'id';
            $type = strtoupper($type) === 'ASC' ? 'ASC' : 'DESC';
            $sql .= " ORDER BY a.{$sort} {$type}";

            $sql .= " LIMIT {$aStart}, {$aLimit}";

            $stmt = $db->prepare($sql);
            $stmt->execute($params);

            $result = $stmt->fetchAll(PDO::FETCH_OBJ);

            return $this->convert_object_to_array($result);
        } catch (PDOException $e) {
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }


    function getOneArticle($id, $lang_code = 'ar')
    {
        try {
            $id = intval($id);
            if ($id <= 0) {
                return ['status' => 400, 'message' => 'Invalid article ID'];
            }

            $db = $this->mDb->connect();

            $sql = "SELECT 
                a.id,
                ac.id_category,
                a.status,
                a.image,
                a.sort,
                a.views,
                a.date_added,
                al.title,
                al.content
            FROM articles a
            LEFT JOIN articles_categories ac ON ac.id_article = a.id
            LEFT JOIN articles_langs al ON al.id_article = a.id AND al.lang_code = :lang_code
            WHERE a.id = :id
            LIMIT 1
        ";

            $stmt = $db->prepare($sql);
            $stmt->execute([
                ':id' => $id,
                ':lang_code' => $lang_code
            ]);

            $article = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$article) {
                return ['status' => 404, 'message' => 'Article not found'];
            }

            // تحديث عدد المشاهدات
            $updateViews = $db->prepare("UPDATE articles SET views = views + 1 WHERE id = :id");
            $updateViews->execute([':id' => $id]);


            // ✅ جلب المسار من الأقسام باستخدام id_cat
            $article['path'] = $this->buildArticlePathFromArticleId($article['id_category'], $lang_code);
            return ['status' => 200, 'data' => $article];
        } catch (PDOException $e) {
            return [
                'status' => 500,
                'message' => 'Database Error: ' . $e->getMessage()
            ];
        }
    }

    function buildArticlePathFromArticleId($category_id, $lang_code = 'ar')
    {
        $db = $this->mDb->connect();
        $path = [];

        while ($category_id != 0) {
            $stmt = $db->prepare("SELECT c.id, c.parent_id, cl.name
            FROM categories c
            LEFT JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = :lang_code
            WHERE c.id = :id
            LIMIT 1
        ");
            $stmt->execute([
                ':id' => $category_id,
                ':lang_code' => $lang_code
            ]);
            $row = $stmt->fetch(PDO::FETCH_ASSOC);

            if (!$row || !$row['name']) break;

            $path[] = $row['name'];
            $category_id = $row['parent_id'];
        }

        return implode(' > ', array_reverse($path));
    }

    function addEditArticle($data)
    {
        try {
            $db = $this->mDb->connect();
            $article_id = isset($data['id']) ? intval($data['id']) : 0;
            $image_name = isset($data['image']) ? trim($data['image']) : '';
            $sort = isset($data['sort']) ? intval($data['sort']) : 0;
            $status = isset($data['status']) ? intval($data['status']) : 1;
            $categories = isset($data['categories']) && is_array($data['categories']) ? $data['categories'] : [];

            if ($article_id == 0 && empty($categories)) {
                return ['status' => 400, 'message' => 'يجب تحديد قسم واحد على الأقل'];
            }

            $db->beginTransaction();

            // ✅ تعديل أو إضافة المقال
            if ($article_id > 0) {
                $stmt = $db->prepare("SELECT COUNT(*) FROM articles WHERE id = ?");
                $stmt->execute([$article_id]);
                if ($stmt->fetchColumn() == 0) {
                    return ['status' => 404, 'message' => 'المقال غير موجود'];
                }

                $stmt = $db->prepare("UPDATE articles SET 
                sort = :sort,
                status = :status
                WHERE id = :id
            ");
                $stmt->execute([
                    ':sort' => $sort,
                    ':status' => $status,
                    ':id' => $article_id
                ]);
            } else {
                $stmt = $db->prepare("INSERT INTO articles (sort, status, date_added) VALUES (:sort, :status, NOW())");
                $stmt->execute([
                    ':sort' => $sort,
                    ':status' => $status
                ]);
                $article_id = $db->lastInsertId();
            }

            // ✅ تحديث الصورة إذا وُجدت فعلاً على السيرفر
            if (!empty($image_name)) {
                $uploadPath = __DIR__ . '/../uploads/articles/';
                $imageFullPath = $uploadPath . $image_name;

                if (file_exists($imageFullPath)) {
                    $stmt = $db->prepare("UPDATE articles SET image = :image WHERE id = :id");
                    $stmt->execute([
                        ':image' => $image_name,
                        ':id' => $article_id
                    ]);
                } else {
                    $db->rollBack();
                    return ['status' => 404, 'message' => 'ملف الصورة غير موجود على السيرفر'];
                }
            }

            // ✅ تحديث الأقسام المرتبطة بالمقال (جدول articles_categories)
            $db->prepare("DELETE FROM articles_categories WHERE id_article = ?")->execute([$article_id]);

            foreach ($categories as $catId) {
                $catId = intval($catId);
                if ($catId > 0) {
                    $stmt = $db->prepare("INSERT INTO articles_categories (id_article, id_category) VALUES (?, ?)");
                    $stmt->execute([$article_id, $catId]);
                }
            }

            // ✅ تحديث اللغات
            if (isset($data['langs']) && is_array($data['langs'])) {
                foreach ($data['langs'] as $lang) {
                    if (empty($lang['lang_code'])) continue;

                    $stmtCheck = $db->prepare("SELECT COUNT(*) FROM articles_langs WHERE id_article = :id AND lang_code = :lang");
                    $stmtCheck->execute([
                        ':id' => $article_id,
                        ':lang' => $lang['lang_code']
                    ]);
                    $exists = $stmtCheck->fetchColumn();

                    if ($exists) {
                        $stmt = $db->prepare("UPDATE articles_langs 
                        SET title = :title, content = :content 
                        WHERE id_article = :id AND lang_code = :lang");
                    } else {
                        $stmt = $db->prepare("INSERT INTO articles_langs 
                        (id_article, lang_code, title, content) 
                        VALUES (:id, :lang, :title, :content)");
                    }

                    $stmt->execute([
                        ':id' => $article_id,
                        ':lang' => $lang['lang_code'],
                        ':title' => isset($lang['title']) ? cleanText($lang['title']) : '',
                        ':content' => isset($lang['content']) ? cleanText($lang['content']) : ''
                    ]);
                }
            }

            $db->commit();
            $isEdit = isset($data['id']) && intval($data['id']) > 0;
            return [
                'status' => true,
                'message' => $isEdit > 0 ? 'تم تعديل المقال بنجاح' : 'تم إضافة المقال بنجاح',
                'article_id' => $article_id
            ];
        } catch (PDOException $e) {
            if ($db->inTransaction()) $db->rollBack();
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }



    // function uploadImageSimple($folder = '')
    // {
    //     if (empty($folder)) {
    //         return ['status' => 0, 'message' => '⚠️ لم يتم تحديد مجلد الرفع'];
    //     }

    //     // استخدم __DIR__ لبناء المسار الكامل الفعلي للمجلد
    //     $basePath = realpath(__DIR__ . '/../../upload') . '/' . $folder . '/';

    //     // تأكد أن المسار تم تحليله بنجاح وأن المجلد موجود
    //     if (!$basePath || !is_dir($basePath)) {
    //         return ['status' => 400, 'message' => '📁 المجلد غير موجود ' ];
    //     }

    //     if (!isset($_FILES['image'])) {
    //         return ['status' => 0, 'message' => '⚠️ لم يتم إرسال ملف الصورة'];
    //     }

    //     $file = $_FILES["image"];
    //     $fileName = basename($file["name"]);
    //     $target_file = $basePath . $fileName;

    //     if (move_uploaded_file($file["tmp_name"], $target_file)) {
    //         return [
    //             'status' => 1,
    //             'message' => '✅ تم رفع الصورة بنجاح',
    //             'file_path' => 'upload/' . $folder . '/' . $fileName // مسار نسبي لتخزينه مثلاً في قاعدة البيانات
    //         ];
    //     } else {
    //         return ['status' => 0, 'message' => '❌ فشل في رفع الصورة'];
    //     }
    // }



    function deleteArticles($ids)
    {
        try {
            $db = $this->mDb->connect();
            $db->beginTransaction();

            if (!is_array($ids) || count($ids) == 0) {
                return ['status' => 400, 'message' => 'Invalid or empty article ID list'];
            }

            // تأكد من IDs أرقام صحيحة
            $ids = array_map('intval', $ids);
            $placeholders = implode(',', array_fill(0, count($ids), '?'));

            // 📸 حذف الصور المرتبطة
            $stmtImgs = $db->prepare("SELECT image FROM articles WHERE id IN ($placeholders)");
            $stmtImgs->execute($ids);
            $images = $stmtImgs->fetchAll(PDO::FETCH_COLUMN);

            $uploadPath = __DIR__ . '/../upload/articles/';
            foreach ($images as $img) {
                if ($img && file_exists($uploadPath . $img)) {
                    unlink($uploadPath . $img);
                }
            }

            // 🗑 حذف الترجمات
            $stmtLangs = $db->prepare("DELETE FROM articles_langs WHERE id_article IN ($placeholders)");
            $stmtLangs->execute($ids);

            // 🗑 حذف المقالات
            $stmtArticles = $db->prepare("DELETE FROM articles WHERE id IN ($placeholders)");
            $stmtArticles->execute($ids);

            $db->commit();

            return [
                'status' => 200,
                'message' => 'Article(s) deleted successfully',
                'deleted_ids' => $ids
            ];
        } catch (PDOException $e) {
            if ($db->inTransaction()) $db->rollBack();
            return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
        }
    }

function getSearchArticles($searchTitle = '', $searchId = '', $searchCategoryId = '', $searchSort = '', $lang_code = 'ar', $aStart = 0, $aLimit = 10, $type = 'DESC', $export_excel = false){
    $type = strtoupper($type);
    if (!in_array($type, ['ASC', 'DESC'])) {
        $type = 'DESC';
    }

    // 🔥 التأكد من أن aStart و aLimit هما أرقام صحيحة
    $aStart = (int) $aStart;
    $aLimit = (int) $aLimit;

    // بناء الاستعلام الرئيسي - تم تحسينه لحل مشكلة البحث بالقسم
    $sql = "SELECT DISTINCT
            a.id,
            a.status,
            a.image,
            a.sort,
            a.date_added,
            COALESCE(al.title, '') as title,
            COALESCE(al.content, '') as content
        FROM articles a
        LEFT JOIN articles_langs al ON (al.id_article = a.id AND al.lang_code = ?)";

    $params = [$lang_code];
    
    // إضافة JOIN للأقسام فقط لو في بحث بالقسم
    $needsCategoryJoin = false;
    if (!empty($searchCategoryId)) {
        $categoryIds = [];
        
        if (is_array($searchCategoryId)) {
            $categoryIds = array_map('intval', array_filter($searchCategoryId, function($val) {
                return !empty($val) && is_numeric($val);
            }));
        } else {
            $searchCategoryId = trim($searchCategoryId);
            if (strpos($searchCategoryId, ',') !== false) {
                $categoryIds = array_map('intval', array_filter(explode(',', $searchCategoryId), function($val) {
                    return !empty(trim($val)) && is_numeric(trim($val));
                }));
            } else {
                if (is_numeric($searchCategoryId)) {
                    $categoryIds[] = intval($searchCategoryId);
                }
            }
        }
        
        if (!empty($categoryIds)) {
            $needsCategoryJoin = true;
            $sql .= " INNER JOIN articles_categories ac ON ac.id_article = a.id";
        }
    }

    $sql .= " WHERE 1 ";

    // معالجة البحث بالعنوان
    if (!empty($searchTitle)) {
        $sql .= " AND al.title LIKE ?";
        $params[] = "%{$searchTitle}%";
    }

    // معالجة البحث بأرقام المقالات
    if (!empty($searchId)) {
        $ids = [];
        
        if (is_array($searchId)) {
            $ids = array_map('intval', array_filter($searchId, function($val) {
                return !empty($val) && is_numeric($val);
            }));
        } else {
            $searchId = trim($searchId);
            if (strpos($searchId, ',') !== false) {
                $ids = array_map('intval', array_filter(explode(',', $searchId), function($val) {
                    return !empty(trim($val)) && is_numeric(trim($val));
                }));
            } else {
                if (is_numeric($searchId)) {
                    $ids[] = intval($searchId);
                }
            }
        }
        
        if (!empty($ids)) {
            if (count($ids) == 1) {
                $sql .= " AND a.id = ?";
                $params[] = $ids[0];
            } else {
                $placeholders = str_repeat('?,', count($ids) - 1) . '?';
                $sql .= " AND a.id IN ($placeholders)";
                $params = array_merge($params, $ids);
            }
        }
    }

    // معالجة البحث بأرقام الأقسام - الحل المحسن
    if ($needsCategoryJoin && !empty($categoryIds)) {
        if (count($categoryIds) == 1) {
            $sql .= " AND ac.id_category = ?";
            $params[] = $categoryIds[0];
        } else {
            $placeholders = str_repeat('?,', count($categoryIds) - 1) . '?';
            $sql .= " AND ac.id_category IN ($placeholders)";
            $params = array_merge($params, $categoryIds);
        }
    }

    // معالجة البحث بأرقام الترتيب
    if (!empty($searchSort)) {
        $sorts = [];
        
        if (is_array($searchSort)) {
            $sorts = array_map('intval', array_filter($searchSort, function($val) {
                return !empty($val) && is_numeric($val);
            }));
        } else {
            $searchSort = trim($searchSort);
            if (strpos($searchSort, ',') !== false) {
                $sorts = array_map('intval', array_filter(explode(',', $searchSort), function($val) {
                    return !empty(trim($val)) && is_numeric(trim($val));
                }));
            } else {
                if (is_numeric($searchSort)) {
                    $sorts[] = intval($searchSort);
                }
            }
        }
        
        if (!empty($sorts)) {
            if (count($sorts) == 1) {
                $sql .= " AND a.sort = ?";
                $params[] = $sorts[0];
            } else {
                $placeholders = str_repeat('?,', count($sorts) - 1) . '?';
                $sql .= " AND a.sort IN ($placeholders)";
                $params = array_merge($params, $sorts);
            }
        }
    }

    $sql .= " ORDER BY a.sort {$type}, a.id {$type}";

    // إضافة LIMIT
    if (!is_null($aStart) && !is_null($aLimit)) {
        $sql .= " LIMIT {$aStart}, {$aLimit}";
    }

    try {
        // 🔍 طباعة الاستعلام للتشخيص (يمكنك حذف هذا لاحقاً)
        error_log("DEBUG SQL: " . $sql);
        error_log("DEBUG PARAMS: " . json_encode($params));
        
        $stmt = $this->mDb->connect()->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // 🔍 طباعة عدد النتائج (يمكنك حذف هذا لاحقاً)
        error_log("DEBUG RESULT COUNT: " . count($result));

        // لو التصدير للإكسل: ضم جميع الأقسام للمقال الواحد
        if ($export_excel && !empty($result)) {
            foreach ($result as &$row) {
                $row['categories'] = $this->getArticleCategories($row['id'], $lang_code);
            }
            return $this->exportArticlesExcel($result);
        }

        foreach ($result as &$row) {
            $row['categories'] = $this->getArticleCategories($row['id'], $lang_code);
        }
        return $this->convert_object_to_array($result);
    } catch (PDOException $e) {
        error_log("SQL ERROR: " . $e->getMessage());
        return ['status' => 500, 'message' => 'Database Error: ' . $e->getMessage()];
    }
}

// دالة إضافية لحساب العدد الإجمالي للمقالات (مفيدة للـ pagination)
function getSearchArticlesCount($searchTitle = '', $searchId = '', $searchCategoryId = '', $searchSort = '', $lang_code = 'ar') {
    $sql = "SELECT COUNT(DISTINCT a.id) as total_count
        FROM articles a
        LEFT JOIN articles_langs al ON (al.id_article = a.id AND al.lang_code = ?)";

    $params = [$lang_code];
    
    // إضافة JOIN للأقسام فقط لو في بحث بالقسم
    $needsCategoryJoin = false;
    if (!empty($searchCategoryId)) {
        $categoryIds = [];
        
        if (is_array($searchCategoryId)) {
            $categoryIds = array_map('intval', array_filter($searchCategoryId, function($val) {
                return !empty($val) && is_numeric($val);
            }));
        } else {
            $searchCategoryId = trim($searchCategoryId);
            if (strpos($searchCategoryId, ',') !== false) {
                $categoryIds = array_map('intval', array_filter(explode(',', $searchCategoryId), function($val) {
                    return !empty(trim($val)) && is_numeric(trim($val));
                }));
            } else {
                if (is_numeric($searchCategoryId)) {
                    $categoryIds[] = intval($searchCategoryId);
                }
            }
        }
        
        if (!empty($categoryIds)) {
            $needsCategoryJoin = true;
            $sql .= " INNER JOIN articles_categories ac ON ac.id_article = a.id";
        }
    }

    $sql .= " WHERE 1 ";

    // معالجة البحث بالعنوان
    if (!empty($searchTitle)) {
        $sql .= " AND al.title LIKE ?";
        $params[] = "%{$searchTitle}%";
    }

    // معالجة البحث بأرقام المقالات
    if (!empty($searchId)) {
        // نفس الكود السابق...
    }

    // معالجة البحث بأرقام الأقسام
    if ($needsCategoryJoin && !empty($categoryIds)) {
        if (count($categoryIds) == 1) {
            $sql .= " AND ac.id_category = ?";
            $params[] = $categoryIds[0];
        } else {
            $placeholders = str_repeat('?,', count($categoryIds) - 1) . '?';
            $sql .= " AND ac.id_category IN ($placeholders)";
            $params = array_merge($params, $categoryIds);
        }
    }

    // معالجة البحث بأرقام الترتيب
    if (!empty($searchSort)) {
        // نفس الكود السابق...
    }

    try {
        $stmt = $this->mDb->connect()->prepare($sql);
        $stmt->execute($params);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return intval($result['total_count']);
    } catch (PDOException $e) {
        error_log("COUNT SQL ERROR: " . $e->getMessage());
        return 0;
    }
}




    private function getArticleCategories($article_id, $lang_code = 'ar')
    {
        $db = $this->mDb->connect();
        $stmt = $db->prepare("
        SELECT c.id, cl.name
        FROM articles_categories ac
        INNER JOIN categories c ON c.id = ac.id_category
        INNER JOIN categories_langs cl ON cl.id_category = c.id AND cl.lang_code = ?
        WHERE ac.id_article = ?
    ");
        $stmt->execute([$lang_code, $article_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }


    public function countArticlesByFilters(
        $searchTitle = '',
        $searchId = '',
        $searchSort = '',
        $lang_code = 'ar',
        $searchCategoryId = ''
    ) {
        $sql = "SELECT COUNT(DISTINCT a.id)
        FROM articles a
        LEFT JOIN articles_langs al ON al.id_article = a.id AND al.lang_code = ?
        LEFT JOIN articles_categories ac ON ac.id_article = a.id
        WHERE 1";

        $params = [$lang_code];

        $hasFilters = false;
        $filteredCats = [];

        if (trim($searchTitle) !== '') {
            $sql .= " AND al.title LIKE ?";
            $params[] = '%' . $searchTitle . '%';
            $hasFilters = true;
        }

        if (is_numeric($searchId) && intval($searchId) > 0) {
            $sql .= " AND a.id = ?";
            $params[] = intval($searchId);
            $hasFilters = true;
        }

        if (trim($searchSort) !== '') {
            $sql .= " AND a.sort = ?";
            $params[] = $searchSort;
            $hasFilters = true;
        }

        if (!empty($searchCategoryId)) {
            if (is_array($searchCategoryId)) {
                $filteredCats = array_filter(array_map('intval', $searchCategoryId));
            } else {
                $filteredCats = array_filter(array_map('intval', explode(',', $searchCategoryId)));
            }

            if (!empty($filteredCats)) {
                $placeholders = implode(',', array_fill(0, count($filteredCats), '?'));
                $sql .= " AND ac.id_category IN ($placeholders)";
                foreach ($filteredCats as $catId) {
                    $params[] = $catId;
                }
                $hasFilters = true;
            }
        }

        try {
            $stmt = $this->mDb->connect()->prepare($sql);
            $stmt->execute($params);
            $count = $stmt->fetchColumn();

            return [
                'status' => 200,
                'count' => intval($count),
                'filtered' => $hasFilters ? true : false
            ];
        } catch (PDOException $e) {
            return [
                'status' => 500,
                'message' => 'Database Error: ' . $e->getMessage()
            ];
        }
    }
}
